<?php

/**
 * Subscriptions Entries List
 *
 * @package    Wp_Events/admin
 * @subpackage Wp_Events/admin/includes
 * @since      1.0.449
 */

// If this file is called directly, abort.
if ( !defined('WPINC') ) {
	die;
}

/**
 * Includes WP_List_Table class from wordpress core.
 */
if ( !class_exists('WP_List_Table') ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

/**
 * New class that extends WP_List_Table, displays form entries
 * and handles user actions.
 *
 * @since 1.0.0
 * 
 * @see WP_List_Table
 */
class Wp_Events_Subscribers_list extends WP_List_Table {

	/**
	 * Name of the table in database.
	 * 
	 * @since 1.0.449
	 * @var string $table_name
	 */
	protected $table_name = 'events_subscribers';

	/**
	 * Class Constructor.
	 * 
	 * Calls constructor from the parent class.
	 *
	 * @since 1.0.449
	 */
	public function __construct()
	{
		parent::__construct([
			'singular' => __( 'Subscribers', 'simple-wp-events' ),
			//singular name of the listed records
			'plural'   => __( 'Subscribers', 'simple-wp-events' ),
			//plural name of the listed records
			'ajax'     => FALSE
			//should this table support ajax?
		]);
	}

	/**
	 * Message to display when no items are available.
	 * 
	 * @since 1.0.449
	 */
	public function no_items() {
		esc_html_e( 'No Subscribers available.', 'simple-wp-events' );
	}

	/**
	 * Define what data to show on each column of the table.
	 *
	 * @param  Array   $item         Data
	 * @param  String  $column_name  - Current column name
	 *
	 * @since 1.0.449
	 * @return Mixed
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			case 'subscriber_firstname':
			case 'subscriber_lastname':
			case 'subscriber_email':
			case 'subscriber_phone':
			case 'subscriber_texting_permission':
				return $item->$column_name;
			case 'time':
				return $item->time_generated;
			case 'id':
				return $item->id;
			default:
				// Safe fallback for unknown columns
				if ( property_exists( $item, $column_name ) ) {
					return esc_html( $item->$column_name );
				}
				return '—'; // Return dash for unknown columns
		}
	}

	/**
	 * Render the bulk edit checkbox.
	 *
	 * @param  array  $item
	 *
	 * @since 1.0.449
	 * @return string
	 */
	function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="bulk-delete[]" value="%s" />',
			$item->id
		);
	}

	/**
	 * Returns an associative array containing the bulk action.
	 *
	 * @return array
	 * @since 1.1.0
	 */
	public function get_bulk_actions() {

		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		if ( isset( $_GET["display"] ) && $_GET["display"] === 'trash' ) {
			return [
				'permanent-delete' => __( 'Delete Permanently', 'simple-wp-events' ),
				'restore'          => __( 'Restore', 'simple-wp-events' ),
			];
		}

		return [
			'bulk-delete' => __( 'Move to Trash', 'simple-wp-events' ),
		];
	}

	/**
	 * Outputs HTML to display bulk actions and filters.
	 * 
	 * @since 1.0.449
	 * @param $which
	 */
	public function display_tablenav( $which ) {
?>
		<input type='hidden' name='_wpnonce' value='<?php echo esc_attr(wp_create_nonce( 'wp_events_entries' )); ?>' />
		<div class="tablenav <?php echo esc_attr( $which ); ?>">
			<?php 
			if ( $this->has_items() ) : 
				?>
				<div class="alignleft actions bulkactions">
					<?php $this->bulk_actions( $which ); ?>
				</div>
				<?php
			endif;
			$this->extra_tablenav( $which );
			$this->pagination( $which );
			?>
			<br class="clear" />
		</div>
<?php
	}

	/**
	 * Handles single row actions of deletion and restore.
	 * 
	 * @param $item
	 * 
	 * @since 1.0.449
	 * @return string
	 */
	function column_ID( $item ) {

		$display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : 'all'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$page    = isset( $_REQUEST['page'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['page'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	
		$actions = [];
	
		$actions['view_entry'] = sprintf(
			'<a href="edit.php?post_type=wp_events&page=wpe_view_entry&entry=%s&tab=subscriptions&display=%s&_wpnonce=%s">%s</a>',
			esc_attr( $item->id ),
			esc_attr( $display ),
			wp_create_nonce( 'wp_events_entries' ),
			__( 'View', 'simple-wp-events' )
		);
	
		if ( $display === 'trash' ) {
			$actions['delete_permanent'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&tab=subscriptions&display=trash&action=%s&bulk-delete[0]=%s&action2=permanent-delete&_wpnonce=%s" onclick="return confirm(\'Are you sure you want to delete item(s)?\');">%s</a>',
				esc_attr( $page ),
				'permanent-delete',
				esc_attr( $item->id ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Delete Permanently', 'simple-wp-events' )
			);
	
			$actions['restore'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&tab=subscriptions&display=trash&action=%s&bulk-delete[0]=%s&action2=restore&_wpnonce=%s">%s</a>',
				esc_attr( $page ),
				'restore',
				esc_attr( $item->id ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Restore', 'simple-wp-events' )
			);
		} else {
			$actions['delete'] = sprintf(
				'<a href="edit.php?post_type=wp_events&page=%s&tab=subscriptions&display=%s&action=%s&bulk-delete[0]=%s&action2=bulk-delete&_wpnonce=%s">%s</a>',
				esc_attr( $page ),
				esc_attr( $display ),
				'bulk-delete',
				esc_attr( $item->id ),
				wp_create_nonce( 'wp_events_entries' ),
				__( 'Move To Trash', 'simple-wp-events' )
			);
		}
	
		return sprintf(
			'%1$s %2$s',
			esc_html( $item->id ),
			$this->row_actions( $actions )
		);
	}
	

	/**
	 * Handles data query and filter, sorting, and pagination.
	 * 
	 * @since 1.0.449
	 */
	public function prepare_items() {

		if ( ! Wp_Events_Db_Actions::wpe_table_exists( $this->table_name ) ) {
			Wp_Events_Db_Actions::add_subscriber_table();
		}

		// Delete all trash entries when empty trash button is clicked.
		if (
			isset( $_GET['emptytrash'] ) &&
			isset( $_REQUEST['_wpnonce'] ) &&
			wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] )), 'wp_events_entries' ) 
		) {
			$this->wpe_empty_trash();
		}

		$search_key = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';

		$this->_column_headers = [
			$this->get_columns(),
			[], // hidden columns
			$this->get_sortable_columns(),
			$this->get_primary_column_name(),
		];

		/** Process bulk action */
		$this->process_bulk_action();


		$per_page     = $this->get_per_page();
		$current_page = $this->get_pagenum();
		$total_items  = $this->record_count();

		if ( $search_key ) {
			$table_data  = $this->get_event_subscribers( -1, $current_page );
			$table_data  = $this->filter_table_data( $table_data, $search_key );
			$total_items = count( $table_data );
			$per_page    = $total_items;
		} else {
			$table_data = $this->get_event_subscribers( $per_page, $current_page );
		}

		$this->set_pagination_args([
			'total_items' => $total_items,
			//WE have to calculate the total number of items
			'per_page'    => $per_page,
			//WE have to determine how many items to show on a page
		]);


		$this->items = $table_data;
	}

	/**
	 * Gets a list of columns.
	 *
	 * The format is:
	 * - `'internal-name' => 'Title'`
	 *
	 * @since 1.0.449
	 *
	 * @return array
	 */
	public function get_columns() {
		return [
			'cb'         					=> '<input type="checkbox">',
			'id'         					=> __('Id', 'simple-wp-events'),
			'subscriber_firstname' 			=> __('First Name', 'simple-wp-events'),
			'subscriber_lastname'  			=> __('Last Name', 'simple-wp-events'),
			'subscriber_email'      		=> __('Email', 'simple-wp-events'),
			'subscriber_phone'      		=> __('Cell Phone', 'simple-wp-events'),
			'subscriber_texting_permission'	=> __('Texting Permission', 'simple-wp-events'),
			'time'       					=> __('Time', 'simple-wp-events'),
		];
	}

	/**
	 * Columns to make sortable.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {
		return [
			'id'       => ['id', TRUE],
			'time'     => ['time_generated', TRUE],
		];
	}

	/**
	 * Process bulk actions of deletion and restore entries.
	 * 
	 * @since 1.0.449
	 */
	public function process_bulk_action() {
		$action = $this->current_action();
	
		if ( in_array( $action, [ 'bulk-delete', 'permanent-delete', 'restore' ], true ) ) {
	
			// Validate nonce and bulk-delete param
			if ( isset( $_REQUEST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field(wp_unslash( $_REQUEST['_wpnonce'] ) ), 'wp_events_entries' ) ) {
	
				if ( isset( $_GET['bulk-delete'] ) ) {

					if ( isset( $_GET['bulk-delete'] ) ) {
						$delete_arr_raw = sanitize_text_field( wp_unslash( $_GET['bulk-delete'] )); 

						// Sanitize each value as an integer
						$delete_arr = array_map( 'intval', (array) $delete_arr_raw );
					}
	
					if ( is_array( $delete_arr ) ) {
						foreach ( $delete_arr as $id ) {
							$id = (int) $id;
							if ( 'bulk-delete' === $action ) {
								$this->delete_restore_entry( $id, WPE_TRASHED );
							} elseif ( 'permanent-delete' === $action ) {
								$this->delete_restore_entry( $id, WPE_DELETED );
							} elseif ( 'restore' === $action ) {
								$this->delete_restore_entry( $id, WPE_ACTIVE );
							}
						}
	
						$no_of_posts = count( $delete_arr );
						switch ( $action ) {
							case 'bulk-delete':
								$message = $no_of_posts . __( ' item(s) moved to the Trash.', 'simple-wp-events' );
								break;
							case 'permanent-delete':
								$message = $no_of_posts . __( ' item(s) permanently deleted.', 'simple-wp-events' );
								break;
							case 'restore':
								$message = $no_of_posts . __( ' item(s) restored from the Trash.', 'simple-wp-events' );
								break;
						}
						$this->wpe_admin_notice( $message );
					}
				}
			} else {
				wp_die( esc_html__( 'Security check failed.', 'simple-wp-events' ) );
			}
		}
	}
	

	/**
	 * Trash/Delete a customer record.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 *
	 * @param  int  $id  customer ID.
	 * @param int $val value to update in status column.
	 * 
	 * @since 1.0.449
	 *
	 * @return bool|int
	 */
	public function delete_restore_entry( $id, $val ) {
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->update(
			"{$wpdb->prefix}$this->table_name",
			['wpe_status' => $val],
			['id' => $id],
			'%d',
			'%d'
		);
	}

	/**
	 * Returns per page items from screen options.
	 * 
	 * @since 1.0.449
	 * 
	 * @return int 
	 */
	public function get_per_page() {
		// get the current user ID
		$user = get_current_user_id();
		// get the current admin screen
		$screen = get_current_screen();
		// retrieve the "per_page" option
		$screen_option = $screen->get_option( 'per_page', 'option' );
		// retrieve the value of the option stored for the current user
		$per_page = get_user_meta( $user, $screen_option, TRUE );
		if ( empty( $per_page ) || $per_page < 1 ) {
			// get the default value if none is set
			$per_page = $screen->get_option( 'per_page', 'default' );
		}

		return $per_page;
	}

	/**
	 * Returns the count of records (to display on screen) in the database.
	 *
	 * @return null|string
	 * @global object $wpdb instantiation of the wpdb class.
	 *
	 * @since 1.1.0
	 */
	public function record_count() {
		global $wpdb;

		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}
	
		if ( isset( $_GET["display"] ) && $_GET["display"] === 'trash' ) {
			$sql = "SELECT COUNT(*) FROM {$wpdb->prefix}$this->table_name WHERE wpe_status = " . WPE_TRASHED;
		} else {
			$sql = "SELECT COUNT(*) FROM {$wpdb->prefix}$this->table_name WHERE wpe_status = " . WPE_ACTIVE;
		}
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		return $wpdb->get_var( $sql );
	}

	/**
	 * Retrieve subscriber's data from the database.
	 *
	 * @param  int    $per_page  from screen options.
	 * @param  int    $page_number
	 *
	 * @return mixed
	 * @global object $wpdb      instantiation of the wpdb class.
	 *
	 * @since 1.1.0
	 */
	public function get_event_subscribers( $per_page = 5, $page_number = 1 ) {

		global $wpdb;
	
		// Check nonce for security
		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}
	
		// Check if 'display' is set and sanitize it
		$display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash( $_GET['display'] ) ) : '';
		if ( $display === 'trash' ) {
			$sql = "SELECT * FROM {$wpdb->prefix}$this->table_name WHERE wpe_status = " . WPE_TRASHED;
		} else {
			$sql = "SELECT * FROM {$wpdb->prefix}$this->table_name WHERE wpe_status = " . WPE_ACTIVE;
		}
	
		// Order by clause sanitization and validation
		if ( ! empty( $_REQUEST['orderby'] ) ) {
			$orderby = sanitize_sql_orderby( wp_unslash($_REQUEST['orderby']) );
			$sql .= ' ORDER BY ' . $orderby;
	
			if ( ! empty( $_REQUEST['order'] ) ) {
				$order = sanitize_sql_orderby( wp_unslash($_REQUEST['order']) );
				$sql .= ' ' . $order;
			} else {
				$sql .= ' ASC';  // Default order is ASC if 'order' is not set
			}
		} else {
			$sql .= ' ORDER BY time_generated DESC';  // Default sorting if no orderby parameter
		}
	
		// Limit and offset sanitization
		if ( -1 !== $per_page ) {
			$sql .= " LIMIT " . absint( $per_page );  // Ensure per_page is a positive integer
	
			$offset = ( $page_number - 1 ) * $per_page;
			$sql .= ' OFFSET ' . absint( $offset );  // Ensure page_number and offset are positive integers
		}
	
		// Direct query with safety measures
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared
		$results = $wpdb->get_results( $sql, OBJECT );
	
		return $results;
	}
	

	/**
	 * Filters the table data based on search key.
	 *
	 * @param array $table_data
	 * @param string $search_key
	 * 
	 * @since 1.0.449
	 * 
	 * @return bool
	 */
	public function filter_table_data( $table_data, $search_key ) {
		return array_values( array_filter( $table_data, function ( $row ) use ( $search_key ) {
			foreach ( $row as $row_val ) {
				if ( stripos( $row_val, $search_key ) !== FALSE ) {
					return TRUE;
				}
			}
		}));
	}

	/**
	 * Displays the list of views (all, trash) available on this table.
	 *
	 * @since 1.0.449
	 */
	public function views() {

		if ( isset( $_GET['_wpnonce'] ) && ! wp_verify_nonce( sanitize_text_field( wp_unslash($_GET['_wpnonce'] ) ), 'wp_events_entries' ) ) {
			wp_die( esc_html__( 'Security check failed', 'simple-wp-events' ) );
		}

		$wpe_current_display = isset( $_GET['display'] ) ? sanitize_text_field( wp_unslash($_GET['display']) ) : 'all';

		$views = array(
			'all' 	=> '<a '. wpe_is_current( $wpe_current_display, 'all' ) .' href="edit.php?post_type=wp_events&page=wp_forms_entries&tab=subscriptions&display=all">All</a>',
			'trash' => '<a '. wpe_is_current( $wpe_current_display, 'trash' ) .' href="edit.php?post_type=wp_events&page=wp_forms_entries&tab=subscriptions&display=trash">Trash</a>'
		);

		/**
		 * Filters the list of available list table views.
		 *
		 * The dynamic portion of the hook name, `$this->screen->id`, refers
		 * to the ID of the current screen.
		 *
		 * @since 1.0.449
		 *
		 * @param string[] $views An array of available list table views.
		 */
		$views = apply_filters( "views_{$this->screen->id}", $views );

		if ( empty( $views ) ) {
			return;
		}

		$this->screen->render_screen_reader_content( 'heading_views' );

		echo "<ul class='subsubsub'>\n";
		$views_count = count( $views );
		$i = 0;
		foreach ( $views as $class => $view ) {
			$i++;
			echo "\t<li class='" . esc_attr( $class ) . "'>" . wp_kses_post( $view );
			if ( $i < $views_count ) {
				echo " |</li>\n";
			} else {
				echo "</li>\n";
			}
		}
		echo '</ul>';
	}

	/**
	 * Extra controls to be displayed between bulk actions and pagination.
	 * 
	 * Displays empty trash button
	 *
	 * @since 1.1.0
	 *
	 * @param string $which
	 */
	protected function extra_tablenav( $which ) {
		if ( $which == "top" ) {
			$this->empty_trash_button();
		}
	}

	/**
	 * Generates custom admin notices on deletion/restore actions
	 * 
	 * @global string $pagenow 
	 *
	 * @param $message
	 * @since 1.1.0
	 */
	public function wpe_admin_notice( $message ) {
		global $pagenow;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( $pagenow == 'edit.php' && isset( $_REQUEST['page'] ) && 'wp_forms_entries' === $_REQUEST['page'] ) {
			 echo '<div class="notice notice-success is-dismissible">
				 <p>'. esc_html( $message ) .'</p>
			 </div>';
		}
	}

	/**
	 * Displays empty trash button if user is on trash tab and there are
	 * entries in the trash.
	 * 
	 * @since 1.1.0
	 */
	public function empty_trash_button() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['display'] ) && $_GET['display'] == 'trash' ) {
			echo '<input type="submit" name="emptytrash" class="button action" value="Empty Trash" 
			onclick="return confirm(\'Are you sure you want to empty the Trash?\');" />';
		}
	}

	/**
	 * Permanently delete all entries from trash tab.
	 * 
	 * @global object $wpdb instantiation of the wpdb class.
	 * 
	 * @since 1.1.0
	 *
	 * @return bool|int
	 */
	public function wpe_empty_trash() {
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		return $wpdb->update(
			"{$wpdb->prefix}$this->table_name",
			[ 'wpe_status' => WPE_DELETED ],
			[ 'wpe_status' => WPE_TRASHED ],
			'%d',
			'%d'
		);
	}
}
